class KeyManager {
    constructor() {
        this.key = null;
        this.isMonitoring = false;
        this.init();
    }

    init() {
        console.log('🔐 KeyManager initialized');
        this.setupSecurityMonitoring();
    }

    // ========== امنیت ==========

    setupSecurityMonitoring() {
        if (this.isMonitoring) return;
        this.isMonitoring = true;

        // 1. محافظت DevTools
        this.monitorDevTools();

        // 2. Auto-Logout
        this.setupAutoLogout();

        // 3. Sync بین تب‌ها
        this.setupTabSync();

        // 4. محافظت Visibility
        this.setupVisibilityCheck();
    }

    monitorDevTools() {
        const threshold = 160;
        let devtoolsWarned = false;

        setInterval(() => {
            const widthDiff = window.outerWidth - window.innerWidth;
            const heightDiff = window.outerHeight - window.innerHeight;

            if (widthDiff > threshold || heightDiff > threshold) {
                if (!devtoolsWarned) {
                    devtoolsWarned = true;
                    console.warn('⚠️ DevTools detected');

                    // اختیاری: logout اجباری
                    if (confirm('هشدار امنیتی: Developer Console شناسایی شد.\nآیا می‌خواهید از سیستم خارج شوید؟')) {
                        this.clearKey();
                        window.location.href = 'index.html';
                    }
                }
            } else {
                devtoolsWarned = false;
            }
        }, 1000);
    }

    setupAutoLogout() {
        let inactivityTimer;
        const TIMEOUT = 30 * 60 * 1000; // 30 دقیقه

        const resetTimer = () => {
            clearTimeout(inactivityTimer);

            inactivityTimer = setTimeout(() => {
                alert('⏰ نشست شما به دلیل عدم فعالیت منقضی شد');
                this.clearKey();
                window.location.href = 'index.html';
            }, TIMEOUT);
        };

        // رویدادهای فعالیت کاربر
        const events = ['mousedown', 'keypress', 'scroll', 'touchstart', 'click'];
        events.forEach(event => {
            document.addEventListener(event, resetTimer, true);
        });

        resetTimer();
        console.log('⏰ Auto-logout timer started (30 min)');
    }

    setupTabSync() {
        window.addEventListener('storage', (e) => {
            if (e.key === 'forceLogout' && e.newValue === 'true') {
                console.warn('🔄 Logout signal received from another tab');
                this.clearKey();
                alert('شما در تب دیگری از سیستم خارج شدید');
                window.location.href = 'index.html';
            }
        });
    }

    setupVisibilityCheck() {
        document.addEventListener('visibilitychange', () => {
            if (document.hidden) {
                console.log('👁️ Tab hidden');
            } else {
                console.log('👁️ Tab visible');
                // بررسی اعتبار کلید
                if (!this.key && sessionStorage.getItem('encKey')) {
                    console.log('🔄 Attempting to reload key...');
                    this.loadKey().catch(() => {
                        alert('خطا در بازیابی کلید. لطفاً دوباره لاگین کنید');
                        this.clearKey();
                        window.location.href = 'index.html';
                    });
                }
            }
        });
    }

    // ========== مدیریت کلید ==========

    async saveKey(key) {
        if (!key) {
            throw new Error('کلید معتبر نیست');
        }

        this.key = key;

        try {
            // Export کلید به صورت raw bytes
            const exported = await crypto.subtle.exportKey('raw', key);
            const keyArray = Array.from(new Uint8Array(exported));

            // ذخیره در sessionStorage
            sessionStorage.setItem('encKey', JSON.stringify(keyArray));

            console.log('✅ Key saved to sessionStorage');
            return true;
        } catch (error) {
            console.error('❌ Error saving key:', error);
            throw new Error('خطا در ذخیره کلید');
        }
    }

    async loadKey() {
        // اگه کلید در RAM هست، بازگردون
        if (this.key) {
            console.log('✅ Key loaded from RAM');
            return this.key;
        }

        // تلاش برای بازیابی از sessionStorage
        const keyData = sessionStorage.getItem('encKey');
        if (!keyData) {
            console.warn('⚠️ No key found in sessionStorage');
            return null;
        }

        try {
            const keyArray = new Uint8Array(JSON.parse(keyData));

            // Import کلید
            this.key = await crypto.subtle.importKey(
                'raw',
                keyArray,
                { name: 'AES-GCM', length: 256 },
                true,
                ['encrypt', 'decrypt']
            );

            console.log('✅ Key loaded from sessionStorage');
            return this.key;
        } catch (error) {
            console.error('❌ Error loading key:', error);
            this.clearKey();
            throw new Error('خطا در بازیابی کلید');
        }
    }

    clearKey() {
        console.log('🗑️ Clearing encryption key...');

        // پاک کردن از RAM
        this.key = null;

        // پاک کردن از sessionStorage
        sessionStorage.removeItem('encKey');
        sessionStorage.removeItem('userId');
        sessionStorage.removeItem('username');
        sessionStorage.removeItem('loggedIn');

        // سیگنال به تب‌های دیگر
        localStorage.setItem('forceLogout', 'true');
        setTimeout(() => localStorage.removeItem('forceLogout'), 1000);

        console.log('✅ Key cleared');
    }

    getKey() {
        return this.key;
    }

    isKeyAvailable() {
        return this.key !== null;
    }
}

// ========== Instance Global ==========
const keyManager = new KeyManager();

// ========== ساخت کلید رمزنگاری ==========
async function deriveKey(password, salt) {
    const encoder = new TextEncoder();
    const passwordData = encoder.encode(password);

    // Import پسورد به عنوان baseKey
    const baseKey = await crypto.subtle.importKey(
        'raw',
        passwordData,
        'PBKDF2',
        false,  // ⭐ baseKey نیازی به extractable نداره
        ['deriveKey']  // فقط برای derive استفاده میشه
    );

    // Derive کلید نهایی با PBKDF2
    return await crypto.subtle.deriveKey(
        {
            name: 'PBKDF2',
            salt: salt,
            iterations: 100000,
            hash: 'SHA-256'
        },
        baseKey,
        { name: 'AES-GCM', length: 256 },
        true,  // ⭐⭐ extractable: true - کلید نهایی باید extractable باشه
        ['encrypt', 'decrypt']
    );
}
async function hashPassword(password, salt) {
    const encoder = new TextEncoder();
    const passwordData = encoder.encode(password);

    const combined = new Uint8Array(passwordData.length + salt.length);
    combined.set(passwordData);
    combined.set(salt, passwordData.length);

    const hashBuffer = await crypto.subtle.digest('SHA-256', combined);
    const hashArray = Array.from(new Uint8Array(hashBuffer));
    return hashArray.map(b => b.toString(16).padStart(2, '0')).join('');
}
