<?php
header('Content-Type: application/json; charset=utf-8');
require_once "config.php";

try {
    session_start();

    $officeCode = $_SESSION['user_code'] ?? null;
    if (!$officeCode) {
        throw new Exception("No office code found in session.");
    }

    $action = $_POST['action'] ?? null;
    if (!$action) {
        throw new Exception("No action provided.");
    }

    switch ($action) {
        case 'add_history':
            addHistory($pdo, $officeCode);
            break;

        case 'delete_history':
            deleteHistory($pdo, $officeCode);
            exit;

        case 'load_histories':
            loadHistories($pdo, $officeCode);
            break;

        default:
            throw new Exception("Invalid action: " . htmlspecialchars($action));
    }

} catch (Throwable $e) {
    echo json_encode([
        'success' => false,
        'error'   => $e->getMessage()
    ]);
}

// === FUNCTIONS ===

function loadHistories($pdo, $officeCode) {
    $stmt = $pdo->prepare("
    SELECT
        id,
        history_code,
        title_encrypted,
        history_type,
        description_encrypted,
        alert_flag
    FROM medical_histories
    WHERE (office_code = :office_code OR office_code = 'DEFAULT')
      AND status != 'DELETED' AND hide_default_for NOT LIKE :pattern
");
    $stmt->execute([':office_code' => $officeCode, ':pattern' => '%"'.$officeCode.'"%']);

    $histories = [];
    while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
        $histories[] = [
            'title'       => decryptData($row['title_encrypted']),
            'history_type'=> $row['history_type'],
            'description' => decryptData($row['description_encrypted']),
            'alert_flag'  => $row['alert_flag'] ? '<span class="badge bg-label-danger">هشدار</span>'
                : '<span class="badge bg-label-success">عادی</span>',
            'actions'     => '<button class="btn btn-sm btn-danger delete-history" data-code="'.$row['history_code'].'">حذف</button>'
        ];
    }

    echo json_encode(['data' => $histories]);
}

function addHistory($pdo, $officeCode) {
    $title       = trim($_POST['add-history-title'] ?? '');
    $historyType = trim($_POST['add-history-type'] ?? '');
    $description = trim($_POST['add-history-description'] ?? '');
    $alertFlag   = isset($_POST['add-history-alert']) ? 1 : 0;

    if (!$title) {
        throw new Exception("عنوان الزامی است.");
    }

    $historyCode = uniqid("MH-") . '-' . substr(base64_encode(hash('sha256', $officeCode, true)), 0, 20);

    $stmt = $pdo->prepare("
        INSERT INTO medical_histories (
            history_code,
            title_encrypted, title_hash,
            history_type,
            description_encrypted, description_hash,
            alert_flag,
            default_for_all,
            hide_default_for,
            office_code
        ) VALUES (?, ?, ?, ?, ?, ?, ?,0,'[]', ?)
    ");

    $stmt->execute([
        $historyCode,
        encryptData($title), hashData($title),
        $historyType,
        encryptData($description), hashData($description),
        $alertFlag,
        $officeCode
    ]);

    echo json_encode([
        'success' => true,
        'message' => 'سابقه پزشکی با موفقیت افزوده شد.'
    ]);
}

function deleteHistory($pdo, $officeCode){
    if (empty($_POST['code'])) {
        echo json_encode(['success' => false, 'error' => 'شناسه سابقه ارسال نشده است']);
        exit;
    }
    $code = trim($_POST['code']);
    try {
        $bla = "q";
        $stmt = $pdo->prepare("
            SELECT *
            FROM medical_histories
            WHERE history_code = :code
            LIMIT 1
        ");
        $stmt->execute([
            ':code'        => $code
        ]);
        $row = $stmt->fetch(PDO::FETCH_ASSOC);

        if($row['default_for_all'] == 1){
            $bla = "b";

            $hided_users = json_decode($row['hide_default_for']);
            $hided_users[] = $officeCode;
            $hided_json = json_encode($hided_users);
            $stmt = $pdo->prepare("
            UPDATE medical_histories SET hide_default_for = :hided_users
            WHERE history_code = :code
        ");
            $stmt->execute([
                ':hided_users' => $hided_json,
                ':code' => $code
            ]);

        }
        else{
            $bla = "d";

            $stmt = $pdo->prepare("
            UPDATE medical_histories SET status = 'DELETED'
            WHERE history_code = :code
              AND office_code = :office_code
        ");
            $stmt->execute([
                ':code'        => $code,
                ':office_code' => $officeCode
            ]);

        }
        if ($stmt->rowCount()) {
            echo json_encode(['success' => true, 'message' => 'سابقه با موفقیت حذف شد.']);
        } else {
            echo json_encode(['success' => false, 'error' => 'سابقه‌ای برای حذف یافت نشد.']);
        }
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'error' => 'خطای پایگاه داده: ' . $bla]);
    }
}