<?php
header('Content-Type: application/json; charset=utf-8');
require_once "config.php";

try {
    session_start();
    $officeCode = $_SESSION['user_code'] ?? null;
    if (!$officeCode) {
        throw new Exception("No office code found in session.");
    }

    $action = $_POST['action'] ?? null;
    if (!$action) {
        throw new Exception("No action provided.");
    }

    switch ($action) {
        case 'add_patient':
            addPatient($pdo, $officeCode);
            break;

        case 'delete_patient':
            deletePatient($pdo, $officeCode);
            break;

        case 'load_patients':
            loadPatients($pdo, $officeCode);
            break;

        case 'get_patient':
            getPatient($pdo, $officeCode);
            break;

        case 'update_patient':
            updatePatient($pdo, $officeCode);
            break;

        default:
            throw new Exception("Invalid action: " . htmlspecialchars($action));
    }

} catch (Throwable $e) {
    echo json_encode([
        'success' => false,
        'error'   => $e->getMessage()
    ]);
}

// ===== FUNCTIONS =====

function loadPatients($pdo, $officeCode) {
    $stmt = $pdo->prepare("
        SELECT
            id,
            case_number_encrypted,
            patient_code,
            patient_firstname_encrypted,
            patient_lastname_encrypted,
            patient_nationalid_encrypted,
            patient_phone_encrypted,
            status,iv,salt,enc_code
        FROM patients
        WHERE status != 'DELETED'
          AND office_code = :office_code
    ");
    $stmt->execute([':office_code' => $officeCode]);
    $patients = [];
    while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
        $patient_encrypted = [
            'patient_firstname_encrypted'  => decryptData($row['patient_firstname_encrypted']),
            'patient_lastname_encrypted'=> decryptData($row['patient_lastname_encrypted']),
            'patient_phone_encrypted'=> decryptData($row['patient_phone_encrypted']),
            'patient_nationalid_encrypted'=> decryptData($row['patient_nationalid_encrypted'])
        ];
        $iv = $row['iv'];
        $salt = $row['salt'];
        $enc_code = $row['enc_code'];
        $tmp_patient = get_decrypted_data($patient_encrypted,$iv,$salt,$enc_code);
        file_put_contents(__DIR__ . '/test2.txt',json_encode($patient_encrypted));
        // تعیین رنگ و متن وضعیت
        if ($row['status'] === 'ACTIVE') {
            $statusHTML = '<span class="badge bg-label-success">فعال</span>';
        } elseif ($row['status'] === 'INCOMPLETE') {
            $statusHTML = '<span class="badge bg-label-warning">تکمیل نشده</span>';
        }

        $patients[] = [
            'case_number' =>decryptData($row['case_number_encrypted']),
            'firstname' => $tmp_patient['patient_firstname_encrypted'],
            'lastname'  => $tmp_patient['patient_lastname_encrypted'],
            'nationalid'=> $tmp_patient['patient_nationalid_encrypted'],
            'phone'     => $tmp_patient['patient_phone_encrypted'],
            'status'    => $statusHTML,
            'actions'   =>
                '<button class="btn btn-sm btn-warning edit-patient" data-code="'.$row['patient_code'].'">ویرایش</button>
         <button class="btn btn-sm btn-danger delete-patient" data-code="'.$row['patient_code'].'">حذف</button>
         <a href="'. ROOT . 'edit-patients?pcode=' . $row['patient_code'] .  '" class="btn btn-sm btn-info complete-patient" data-code="'.$row['patient_code'].'">تکمیل بیمار</a>
         <a href="'. ROOT . 'appointments?pcode=' . $row['patient_code'] .  '" class="btn btn-sm btn-primary schedule-appointment" data-code="'.$row['patient_code'].'">نوبت دهی</a>
         <a href="'. ROOT . 'cases/' . $row['patient_code'] .  '" class="btn btn-sm btn-success treatment-plan" data-code="'.$row['patient_code'].'">مشاهده پرونده</a>',
            'barcode'   => 'jhbjbj'
        ];
    }

    echo json_encode(['data' => $patients]);
}

function addPatient($pdo, $officeCode) {
    $firstName  = trim($_POST['add-patient-firstname'] ?? '');
    $lastName   = trim($_POST['add-patient-lastname'] ?? '');
    $nationalId = trim($_POST['add-patient-nationalid'] ?? '');
    $phone      = trim($_POST['add-patient-phone'] ?? '');
    $fullName    = $firstName . ' ' . $lastName;

    $payload = [];
    $payload['add-patient-firstname'] = trim($_POST['add-patient-firstname'] ?? '');
    $payload['add-patient-lastname']  = trim($_POST['add-patient-lastname'] ?? '');
    $payload['add-patient-fullname']  = trim($_POST['add-patient-firstname'] . ' ' . $_POST['add-patient-lastname'] ?? '');
    $payload['add-patient-phone']     = trim($_POST['add-patient-phone'] ?? '');
    $payload['add-patient-nationalid']= trim($_POST['add-patient-nationalid'] ?? '');

    $enc = get_encrypted_data(json_encode($payload));
    $enc_data = $enc['encrypted_data'];
    $firstName_enc  = trim($enc_data['add-patient-firstname'] ?? '');
    $lastName_enc   = trim($enc_data['add-patient-lastname'] ?? '');
    $fullName_enc   = trim($enc_data['add-patient-fullname'] ?? '');
    $nationalId_enc = trim($enc_data['add-patient-nationalid'] ?? '');
    $phone_enc      = trim($enc_data['add-patient-phone'] ?? '');
    $iv = $enc['iv'];
    $salt = $enc['salt'];
    $enc_code = $enc['code'];
    if ($firstName === '' || $lastName === '') {
        throw new Exception("نام و نام خانوادگی الزامی است.");
    }

    // --- Generate next numeric case number (PHP-side) ---
    $stmt = $pdo->prepare("
        SELECT case_number_encrypted 
        FROM patients 
        WHERE office_code = :office
    ");
    $stmt->execute([':office' => $officeCode]);
    $maxCase = 0;
    while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
        $decrypted = decryptData($row['case_number_encrypted']); // same as encryptData's reverse
        if (ctype_digit($decrypted) && (int)$decrypted > $maxCase) {
            $maxCase = (int)$decrypted;
        }
    }
    $nextCaseNumber = $maxCase + 1;

    // --- Other generated values ---
    $patientCode = uniqid("PAT-") . '-' . substr(base64_encode(hash('sha256', $officeCode)), 0, 20);

    // --- Insert ---
    $stmt = $pdo->prepare("
        INSERT INTO patients (
    patient_code,
    patient_firstname_encrypted, patient_firstname_hash,
    patient_lastname_encrypted, patient_lastname_hash,
    patient_fullname_encrypted, patient_fullname_hash,
    patient_nationalid_encrypted, patient_nationalid_hash,
    patient_phone_encrypted, patient_phone_hash,
    case_number_encrypted, case_number_hash,
    status,iv,salt,enc_code,
    office_code
) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 'ACTIVE', ?, ?, ?, ?)
    ");

    $stmt->execute([
        $patientCode,
        encryptData($firstName_enc), hashData($firstName),
        encryptData($lastName_enc),  hashData($lastName),
        encryptData($fullName_enc),  hashData($fullName),
        encryptData($nationalId_enc),hashData($nationalId),
        encryptData($phone_enc),     hashData($phone),
        encryptData((string)$nextCaseNumber), hashData((string)$nextCaseNumber),
        $iv,$salt,$enc_code,
        $officeCode
    ]);

    echo json_encode(['success' => true, 'message' => 'بیمار با موفقیت افزوده شد.']);
}

function deletePatient($pdo, $officeCode) {
    if (empty($_POST['code'])) {
        echo json_encode(['success' => false, 'error' => 'شناسه بیمار ارسال نشده است']);
        return;
    }

    $code = trim($_POST['code']);
    $stmt = $pdo->prepare("
        UPDATE patients
        SET status = 'DELETED'
        WHERE patient_code = :code
          AND office_code = :office_code
    ");
    $stmt->execute([
        ':code'        => $code,
        ':office_code' => $officeCode
    ]);

    if ($stmt->rowCount()) {
        echo json_encode(['success' => true, 'message' => 'بیمار با موفقیت حذف شد.']);
    } else {
        echo json_encode(['success' => false, 'error' => 'بیماری برای حذف یافت نشد.']);
    }
}

function getPatient($pdo, $officeCode) {
    if (empty($_POST['code'])) {
        echo json_encode(['success' => false, 'error' => 'شناسه معتبر نمی‌باشد']);
        return;
    }

    $code = trim($_POST['code']);
    $stmt = $pdo->prepare("
        SELECT patient_code,
               patient_firstname_encrypted,
               patient_lastname_encrypted,
               patient_nationalid_encrypted,
               patient_phone_encrypted,iv,salt,enc_code
        FROM patients
        WHERE patient_code = :code
          AND office_code = :office_code
          AND status != 'DELETED'
        LIMIT 1
    ");
    $stmt->execute([
        ':code'        => $code,
        ':office_code' => $officeCode
    ]);
    $row = $stmt->fetch(PDO::FETCH_ASSOC);
    if ($row) {
        $patient_encrypted = [
            'patient_firstname_encrypted' => decryptData($row['patient_firstname_encrypted']),
            'patient_lastname_encrypted'  => decryptData($row['patient_lastname_encrypted']),
            'patient_nationalid_encrypted'=> decryptData($row['patient_nationalid_encrypted']),
            'patient_phone_encrypted'=> decryptData($row['patient_phone_encrypted'])
        ];
        $iv = $row['iv'];
        $salt = $row['salt'];
        $enc_code = $row['enc_code'];
        $tmp_patient = get_decrypted_data($patient_encrypted,$iv,$salt,$enc_code);
        echo json_encode([
            'success' => true,
            'patient' => [
                'id'         => $row['patient_code'],
                'firstname'  => $tmp_patient['patient_firstname_encrypted'],
                'lastname'   => $tmp_patient['patient_lastname_encrypted'],
                'nationalid' => $tmp_patient['patient_nationalid_encrypted'],
                'phone'      => $tmp_patient['patient_phone_encrypted']
            ]
        ]);
    } else {
        echo json_encode(['success' => false, 'error' => 'بیمار یافت نشد.']);
    }
}

function updatePatient($pdo, $officeCode) {
    if (empty($_POST['code'])) {
        echo json_encode(['success' => false, 'error' => 'کد بیمار ارسال نشده است.']);
        return;
    }

    $code       = trim($_POST['code']);
    $firstName  = trim($_POST['firstname'] ?? '');
    $lastName   = trim($_POST['lastname'] ?? '');
    $nationalId = trim($_POST['nationalid'] ?? '');
    $phone      = trim($_POST['phone'] ?? '');

    $payload = [];
    $payload['firstname'] = trim($_POST['firstname'] ?? '');
    $payload['lastname']  = trim($_POST['lastname'] ?? '');
    $payload['fullname']  = trim($_POST['firstname'] . ' ' . $_POST['lastname'] ?? '');
    $payload['phone']     = trim($_POST['phone'] ?? '');
    $payload['nationalid']= trim($_POST['nationalid'] ?? '');

    $enc = get_encrypted_data(json_encode($payload));
    $enc_data = $enc['encrypted_data'];
    $iv = $enc['iv'];
    $salt = $enc['salt'];
    $enc_code = $enc['code'];

    $firstName_enc  = $enc_data['firstname'] ?? '';
    $lastName_enc   = $enc_data['lastname'] ?? '';
    $fullName_enc   = $enc_data['fullname'] ?? '';
    $nationalId_enc = $enc_data['nationalid'] ?? '';
    $phone_enc      = $enc_data['phone'] ?? '';

    if ($firstName === '' || $lastName === '' || $nationalId === '' || $phone === '') {
        echo json_encode(['success' => false, 'error' => 'تمام فیلدها الزامی هستند.']);
        return;
    }

    $stmt = $pdo->prepare("
        UPDATE patients
        SET patient_firstname_encrypted = :firstname,
            patient_lastname_encrypted  = :lastname,
            patient_fullname_encrypted  = :fullname,
            patient_nationalid_encrypted= :nationalid,
            patient_phone_encrypted     = :phone,
            iv = :iv,
            salt = :salt,
            enc_code = :enc_code
        WHERE patient_code = :code
          AND office_code  = :office_code
          AND status != 'DELETED'
    ");
    $updated = $stmt->execute([
        ':firstname'   => encryptData($firstName_enc),
        ':lastname'    => encryptData($lastName_enc),
        ':fullname'    => encryptData($fullName_enc),
        ':nationalid'  => encryptData($nationalId_enc),
        ':phone'       => encryptData($phone_enc),
        ':iv' => $iv,
        ':salt' => $salt,
        ':enc_code' => $enc_code,
        ':code'        => $code,
        ':office_code' => $officeCode
    ]);

    if ($updated && $stmt->rowCount() > 0) {
        echo json_encode(['success' => true, 'message' => 'اطلاعات بیمار به‌روز شد.']);
    } else {
        echo json_encode(['success' => false, 'error' => 'بیمار یافت نشد یا تغییری اعمال نشد.']);
    }
}
