<?php
header('Content-Type: application/json; charset=utf-8');
require_once "config.php";

try {
    session_start();

    $officeCode = $_SESSION['office_code'] ?? $_SESSION['user_code'] ?? null;
    if (!$officeCode) {
        throw new Exception("No office code found in session.");
    }

    $action = $_POST['action'] ?? null;
    if (!$action) {
        throw new Exception("No action provided.");
    }

    switch ($action) {
        /* === SERVICES === */
        case 'load_services':
            loadServices($pdo, $officeCode);
            break;

        case 'add_service':
            addService($pdo, $officeCode);
            break;

        case 'delete_service':
            deleteService($pdo, $officeCode);
            exit;

        case 'get_service':
            getService($pdo, $officeCode);
            exit;

        case 'update_service':
            updateService($pdo, $officeCode);
            exit;

        /* === CATEGORIES === */
        case 'load_service_categories':
            loadCategories($pdo, $officeCode);
            break;

        case 'add_service_category':
            addCategory($pdo, $officeCode);
            break;

        case 'delete_service_category':
            deleteCategory($pdo, $officeCode);
            exit;

        case 'get_service_category':
            getCategory($pdo, $officeCode);
            exit;

        case 'update_service_category':
            updateCategory($pdo, $officeCode);
            exit;
        case 'fetch_categories_list':
            fetchCategoriesList($pdo, $officeCode);
            exit;

        default:
            throw new Exception("Invalid action: " . htmlspecialchars($action));
    }

} catch (Throwable $e) {
    echo json_encode([
        'success' => false,
        'error'   => $e->getMessage()
    ]);
}

/* === SERVICES FUNCS === */

function loadServices($pdo, $officeCode) {
    // 1. Get all active services for this office
    $stmt = $pdo->prepare("
        SELECT service_code, service_title, service_category_code
        FROM services
        WHERE status = 'ACTIVE' AND COALESCE(hide_default_for, '') NOT LIKE :pattern
          AND (office_code = :office_code OR office_code = 'DEFAULT')
    ");
    $stmt->execute([':office_code' => $officeCode, ':pattern' => '%"'.$officeCode.'"%']);
    $services = $stmt->fetchAll(PDO::FETCH_ASSOC);

    // 2. Get all active categories for this office into a key/value map
    $catStmt = $pdo->prepare("
        SELECT service_category_code, category_title
        FROM service_categories
        WHERE (office_code = :office_code OR office_code = 'DEFAULT')
          AND status = 'ACTIVE' AND COALESCE(hide_default_for, '') NOT LIKE :pattern
    ");
    $catStmt->execute([':office_code' => $officeCode, ':pattern' => '%"'.$officeCode.'"%']);
    $categoryMap = [];
    foreach ($catStmt->fetchAll(PDO::FETCH_ASSOC) as $cat) {
        $categoryMap[$cat['service_category_code']] = $cat['category_title'];
    }

    // 2. Get all active categories for this office into a key/value map
    $tariffStmt = $pdo->prepare("
        SELECT *
        FROM tariffs
        WHERE office_code = :office_code
          AND status = 'ACTIVE'
    ");
    $tariffStmt->execute([':office_code' => $officeCode]);
    $tariffs = [];
    foreach ($tariffStmt->fetchAll(PDO::FETCH_ASSOC) as $tariff) {
        $tariffs[$tariff['for_code']] = $tariff;
    }

    // 3. Build response array, looking up category title from map
    $data = [];
    foreach ($services as $srv) {
        $categoryTitle = $categoryMap[$srv['service_category_code']] ?? '—';
        $tariff = $tariffs[$srv['service_code']] ?? [];
        if(empty($tariff)){
            $price = 'تعریف نشده';
            $tariffType_output = 'تعریف نشده';
        }
        else{
            $price = $tariffs[$srv['service_code']]['price'] ?? '0';
            $price = thousand_sepr($price) . ' تومان';
            $tariffType =$tariffs[$srv['service_code']]['type'] ?? '-';
            $tariffType_output = '-';
            if($tariffType == "COUNTABLE"){
                $tariffType_output = 'لحاظ می شود.';
            }
            else{
                if($tariffType == "NOTCOUNTABLE"){
                    $tariffType_output = 'لحاظ نمی شود.';
                }
            }
        }
        $data[] = [
            'service_title'  => htmlspecialchars($srv['service_title']),
            'category_title' => htmlspecialchars($categoryTitle),
            'price' => htmlspecialchars($price),
            'type' => htmlspecialchars($tariffType_output),
            'status'         => '<span class="badge bg-label-success">فعال</span>',
            'actions'        => '<button class="btn btn-sm btn-warning edit-service" data-code="'.$srv['service_code'].'">ویرایش</button>
                                  <button class="btn btn-sm btn-danger delete-service" data-code="'.$srv['service_code'].'">حذف</button>'
        ];
    }

    echo json_encode(['data' => $data]);
}

function addService($pdo, $officeCode) {
    $title = trim($_POST['add-service-title'] ?? '');
    $cat   = trim($_POST['add-service-category'] ?? '');
    $add_tariff = trim($_POST['add-service-tariff'] ?? '');
    $tariff_price = trim($_POST['add-tariff-price'] ?? '');
    $tariff_price = str_replace(',', '', $tariff_price);
    if(isset($_POST['add-tariff-type'])){
        $tariff_type = trim($_POST['add-tariff-type']) ?? 'off';
    }
    else{
        $tariff_type = "off";
    }
    if ($title === '' || $cat === '') {
        echo json_encode(['success' => false, 'error' => 'تمام فیلدها الزامی هستند.']);
        return;
    }
    if($add_tariff === 'NA') {
    }
    else{
        if ($tariff_price === '' || $tariff_price === '0' || $tariff_price === 0) {
            echo json_encode(['success' => false, 'error' => 'در صورت افزودن هزینه باید مبلغ سرویس نیز مشخص باشد.']);
            return;
        }
    }
    $code = uniqid("SRV-") . '-' . substr(base64_encode(hash('sha256', $officeCode)), 0, 20);
    $stmt = $pdo->prepare("
        INSERT INTO services (service_code, service_title, service_category_code, office_code) 
        VALUES (?, ?, ?, ?)
    ");
    $stmt->execute([$code, $title, $cat, $officeCode]);
    if($add_tariff === 'NA') {
    }
    else{
        $type = "NOTCOUNTABLE";
        if($tariff_type === 'on'){
            $type = "COUNTABLE";
        }
        $tariff_code = uniqid("TRF-") . '-' . substr(base64_encode(hash('sha256', $officeCode)), 0, 20);
        $stmt = $pdo->prepare("
        INSERT INTO tariffs (tariff_code,office_code, for_type, for_code, type,price) 
        VALUES (?,?,'services',?, ?, ?)
    ");
        $stmt->execute([$tariff_code,$officeCode, $code, $type,$tariff_price]);
    }
    echo json_encode(['success' => true, 'message' => 'خدمت با موفقیت افزوده شد.']);
}

function deleteService($pdo, $officeCode) {
    $code = trim($_POST['code'] ?? '');
    if ($code === '') {
        echo json_encode(['success'=>false,'error'=>'کد خدمت ارسال نشده']);
        return;
    }
    $stmt = $pdo->prepare("
        SELECT service_code, office_code,hide_default_for FROM services
        WHERE service_code=:code AND status!='DELETED'
        LIMIT 1
    ");
    $stmt->execute([':code'=>$code]);
    $row = $stmt->fetch(PDO::FETCH_ASSOC);
    if ($row){
        if($row['office_code'] === "DEFAULT"){
            $hide_for_json = $row['hide_default_for'];
            $hide_for_array = json_decode($hide_for_json, true);
            $hide_for_array[] = $officeCode;
            $stmt = $pdo->prepare("
        UPDATE services SET hide_default_for=:hide_for
        WHERE service_code=:code
    ");
            $stmt->execute([':hide_for'=>json_encode($hide_for_array), ':code'=>$code]);
        }
        else{
            $stmt = $pdo->prepare("
        UPDATE services SET status='DELETED' 
        WHERE service_code=:code AND office_code=:office_code
    ");
            $stmt->execute([':code'=>$code, ':office_code'=>$officeCode]);
        }
    }
    echo json_encode(['success'=> ($stmt->rowCount()>0), 'message'=>'حذف شد', 'error'=>'حذف نشد']);
}

function getService($pdo, $officeCode) {
    $code = trim($_POST['code'] ?? '');
    $stmt = $pdo->prepare("
        SELECT * FROM tariffs
        WHERE for_code=:code AND office_code = :office_code AND status!='DELETED'
        LIMIT 1
    ");
    $stmt->execute([':code'=>$code, ':office_code'=>$officeCode]);
    $row_tariff = $stmt->fetch(PDO::FETCH_ASSOC);
    $stmt = $pdo->prepare("
        SELECT service_code, service_title, service_category_code FROM services
        WHERE service_code=:code AND (office_code = :office_code OR office_code = 'DEFAULT') AND status!='DELETED'
        LIMIT 1
    ");
    $stmt->execute([':code'=>$code, ':office_code'=>$officeCode]);
    $row = $stmt->fetch(PDO::FETCH_ASSOC);
    if ($row)
        echo json_encode(['success'=>true, 'service'=>$row,'tariff'=>$row_tariff]);
    else
        echo json_encode(['success'=>false, 'error'=>'خدمت یافت نشد']);
}

function updateService($pdo, $officeCode) {
    $code = trim($_POST['code'] ?? '');
    $title = trim($_POST['service_title'] ?? '');
    $cat   = trim($_POST['service_category_code'] ?? '');
    $add_tariff = trim($_POST['service_add_tariff'] ?? '');
    $tariff_price = trim($_POST['service_tariff_price'] ?? '');
    $tariff_price = str_replace(',', '', $tariff_price);
    if(isset($_POST['service_add_tariff'])){
        $tariff_type = trim($_POST['service_tariff_type']) ?? 'off';
    }
    else{
        $tariff_type = "off";
    }
    if($add_tariff === 'NA') {
    }
    else{
        if ($tariff_price === '' || $tariff_price === '0' || $tariff_price === 0) {
            echo json_encode(['success' => false, 'error' => 'در صورت افزودن هزینه باید مبلغ سرویس نیز مشخص باشد.']);
            return;
        }
    }

    if ($code===''||$title===''||$cat==='') {
        echo json_encode(['success'=>false,'error'=>'تمام فیلدها الزامی هستند.']);
        return;
    }
    $stmt = $pdo->prepare("
        SELECT service_code, office_code,hide_default_for FROM services
        WHERE service_code=:code AND status!='DELETED'
        LIMIT 1
    ");
    $stmt->execute([':code'=>$code]);
    $row = $stmt->fetch(PDO::FETCH_ASSOC);
    if ($row) {
        if ($row['office_code'] === "DEFAULT") {
            $hide_for_json = $row['hide_default_for'];
            $hide_for_array = json_decode($hide_for_json, true);
            $hide_for_array[] = $officeCode;
            $stmt = $pdo->prepare("
        UPDATE services SET hide_default_for=:hide_for
        WHERE service_code=:code
    ");
            $stmt->execute([':hide_for'=>json_encode($hide_for_array), ':code'=>$code]);
            $code = uniqid("SRV-") . '-' . substr(base64_encode(hash('sha256', $officeCode)), 0, 20);
            $stmt = $pdo->prepare("
        INSERT INTO services (service_code, service_title, service_category_code, office_code) 
        VALUES (?, ?, ?, ?)
    ");
            $stmt->execute([$code, $title, $cat, $officeCode]);
            if($add_tariff === 'NA') {
            }
            else{
                $type = "NOTCOUNTABLE";
                if($tariff_type === 'on'){
                    $type = "COUNTABLE";
                }
                $tariff_code = uniqid("TRF-") . '-' . substr(base64_encode(hash('sha256', $officeCode)), 0, 20);
                $stmt = $pdo->prepare("
        INSERT INTO tariffs (tariff_code,office_code, for_type, for_code, type,price) 
        VALUES (?,?,'services',?, ?, ?)
    ");
                $stmt->execute([$tariff_code,$officeCode, $code, $type,$tariff_price]);
            }
        }
        else{
            $stmt = $pdo->prepare("
        UPDATE services SET service_title=:title, service_category_code=:cat
        WHERE service_code=:code AND office_code=:office_code AND status!='DELETED'
    ");
            $stmt->execute([
                ':title'=>$title, ':cat'=>$cat, ':code'=>$code, ':office_code'=>$officeCode
            ]);
            if($add_tariff === 'NA') {
            }
            else{
                $type = "NOTCOUNTABLE";
                if($tariff_type === 'on'){
                    $type = "COUNTABLE";
                }
                $tariffStmt = $pdo->prepare("
        SELECT *
        FROM tariffs
        WHERE office_code = :office_code
          AND status = 'ACTIVE'
    ");
                $tariffStmt->execute([':office_code' => $officeCode]);
                $tariffs = [];
                foreach ($tariffStmt->fetchAll(PDO::FETCH_ASSOC) as $tariff) {
                    $tariffs[$tariff['for_code']] = $tariff;
                }
                if($tariffs[$code]){
                    $tariff_code = $tariffs[$code]['tariff_code'];
                    $stmt = $pdo->prepare("
        UPDATE tariffs SET for_type='services', for_code=?, type=?, price=?, office_code=? WHERE tariff_code=?
    ");
                    $stmt->execute([$code, $type,$tariff_price, $officeCode, $tariff_code]);
                }
                else{
                    $tariff_code = uniqid("TRF-") . '-' . substr(base64_encode(hash('sha256', $officeCode)), 0, 20);
                    $stmt = $pdo->prepare("
        INSERT INTO tariffs (tariff_code,office_code, for_type, for_code, type,price) 
        VALUES (?,?,'services',?, ?, ?)
    ");
                    $stmt->execute([$tariff_code,$officeCode, $code, $type,$tariff_price]);
                }
            }
        }
    }
    echo json_encode(['success'=>($stmt->rowCount()>0), 'message'=>'بروزرسانی شد', 'error'=>'تغییری انجام نشد']);
}

/* === CATEGORY FUNCS === */

function loadCategories($pdo, $officeCode) {
    $stmt = $pdo->prepare("
        SELECT service_category_code, category_title
        FROM service_categories
        WHERE (office_code = :office_code OR office_code = 'DEFAULT')
          AND status = 'ACTIVE' AND COALESCE(hide_default_for, '') NOT LIKE :pattern
    ");
    $stmt->execute([':office_code'=>$officeCode,':pattern'=>'%'.$officeCode.'%']);
    $data = [];
    while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
        $data[] = [
            'category_title' => htmlspecialchars($row['category_title']),
            'status'         => '<span class="badge bg-label-success">فعال</span>',
            'actions'        => '<button class="btn btn-sm btn-warning edit-category" data-code="'.$row['service_category_code'].'">ویرایش</button>
                                  <button class="btn btn-sm btn-danger delete-category" data-code="'.$row['service_category_code'].'">حذف</button>'
        ];
    }
    echo json_encode(['data'=>$data]);
}

function addCategory($pdo, $officeCode) {
    $title = trim($_POST['add-category-title'] ?? '');
    if ($title === '') {
        echo json_encode(['success'=>false,'error'=>'عنوان الزامی است']);
        return;
    }
    $code = uniqid("CAT-") . '-' . substr(base64_encode(hash('sha256', $officeCode)), 0, 20);
    $stmt = $pdo->prepare("
        INSERT INTO service_categories (service_category_code, category_title, office_code) 
        VALUES (?, ?, ?)
    ");
    $stmt->execute([$code, $title, $officeCode]);
    echo json_encode(['success'=>true,'message'=>'دسته‌بندی افزوده شد']);
}

function deleteCategory($pdo, $officeCode) {
    $code = trim($_POST['code'] ?? '');
    if ($code === '') {
        echo json_encode(['success'=>false,'error'=>'کد دسته‌بندی ارسال نشده']);
        return;
    }
    $stmt = $pdo->prepare("
        SELECT service_category_code, office_code,hide_default_for FROM service_categories
        WHERE service_category_code=:code AND status!='DELETED'
        LIMIT 1
    ");
    $stmt->execute([':code'=>$code]);
    $row = $stmt->fetch(PDO::FETCH_ASSOC);
    if ($row){
        if($row['office_code'] === "DEFAULT"){
            $hide_for_json = $row['hide_default_for'];
            $hide_for_array = json_decode($hide_for_json, true);
            $hide_for_array[] = $officeCode;
            $stmt = $pdo->prepare("
        UPDATE service_categories SET hide_default_for=:hide_for
        WHERE service_category_code=:code
    ");
            $stmt->execute([':hide_for'=>json_encode($hide_for_array), ':code'=>$code]);
        }
        else{
            $stmt = $pdo->prepare("
        UPDATE service_categories SET status='DELETED' 
        WHERE service_category_code=:code AND office_code=:office_code
    ");
            $stmt->execute([':code'=>$code, ':office_code'=>$officeCode]);
        }
    }
    echo json_encode(['success'=>($stmt->rowCount()>0),'message'=>'حذف شد','error'=>'حذف نشد']);
}

function getCategory($pdo, $officeCode) {
    $code = trim($_POST['code'] ?? '');
    $stmt = $pdo->prepare("
        SELECT service_category_code, category_title
        FROM service_categories
        WHERE service_category_code=:code AND (office_code = :office_code OR office_code = 'DEFAULT') AND status!='DELETED'
        LIMIT 1
    ");
    $stmt->execute([':code'=>$code, ':office_code'=>$officeCode]);
    $row = $stmt->fetch(PDO::FETCH_ASSOC);
    if ($row)
        echo json_encode(['success'=>true,'category'=>$row]);
    else
        echo json_encode(['success'=>false,'error'=>'دسته‌بندی یافت نشد']);
}

function updateCategory($pdo, $officeCode) {
    $code = trim($_POST['code'] ?? '');
    $title = trim($_POST['category_title'] ?? '');
    if ($code===''||$title==='') {
        echo json_encode(['success'=>false,'error'=>'تمام فیلدها الزامی هستند']);
        return;
    }

    $stmt = $pdo->prepare("
        SELECT service_category_code, office_code,hide_default_for FROM service_categories
        WHERE service_category_code=:code AND status!='DELETED'
        LIMIT 1
    ");
    $stmt->execute([':code'=>$code]);
    $row = $stmt->fetch(PDO::FETCH_ASSOC);
    if ($row){
        if($row['office_code'] === "DEFAULT"){
            $hide_for_json = $row['hide_default_for'];
            $hide_for_array = json_decode($hide_for_json, true);
            $hide_for_array[] = $officeCode;
            $stmt = $pdo->prepare("
        UPDATE service_categories SET hide_default_for=:hide_for
        WHERE service_category_code=:code
    ");
            $stmt->execute([':hide_for'=>json_encode($hide_for_array), ':code'=>$code]);
            $code = uniqid("CAT-") . '-' . substr(base64_encode(hash('sha256', $officeCode)), 0, 20);
            $stmt = $pdo->prepare("
        INSERT INTO service_categories (service_category_code, category_title, office_code) 
        VALUES (?, ?, ?)
    ");
            $stmt->execute([$code, $title, $officeCode]);
        }
        else{
            $stmt = $pdo->prepare("
        UPDATE service_categories SET category_title=:title
        WHERE service_category_code=:code AND office_code=:office_code AND status!='DELETED'
    ");
            $stmt->execute([':title'=>$title, ':code'=>$code, ':office_code'=>$officeCode]);
        }
    }
    echo json_encode(['success'=>($stmt->rowCount()>0),'message'=>'بروزرسانی شد','error'=>'تغییری انجام نشد']);
}
function fetchCategoriesList($pdo, $officeCode) {
    $stmt = $pdo->prepare("
        SELECT service_category_code, category_title
        FROM service_categories
        WHERE (office_code = :office_code OR office_code = 'DEFAULT')
          AND status = 'ACTIVE' AND COALESCE(hide_default_for, '') NOT LIKE :pattern
    ");
    $stmt->execute([':office_code'=>$officeCode,':pattern'=>'%'.$officeCode.'%']);
    $categories = [];
    while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
        $categories[] = [
            'code'  => $row['service_category_code'],
            'title' => $row['category_title']
        ];
    }
    echo json_encode(['success'=>true,'categories'=>$categories]);
}